import cv2
import base64
import numpy as np
from scipy.spatial import distance
from expiringdict import ExpiringDict

from edge_engine.common.logsetup import logger
from scripts.utils.infocenter import MongoLogger
from yolov5processor.infer import ExecuteInference
from scripts.utils.edge_utils import get_extra_fields
from edge_engine.ai.model.modelwraper import ModelWrapper
from scripts.utils.centroidtracker import CentroidTracker
from scripts.common.constants import JanusDeploymentConstants
from scripts.utils.image_utils import draw_circles_on_frame, resize_to_64_64

import cv2
import base64
import datetime
import numpy as np

from collections import deque
from expiringdict import ExpiringDict
from sklearn.utils.linear_assignment_ import linear_assignment

from edge_engine.common.logsetup import logger
from edge_engine.ai.model.modelwraper import ModelWrapper

from scripts.utils.tracker import Tracker
from scripts.utils.helpers import box_iou2
from scripts.utils.edge_utils import Utilities
from scripts.utils.infocenter import MongoLogger
from scripts.utils.model_tracker import ModelCountTracker
from scripts.common.constants import JanusDeploymentConstants

from yolov5processor.infer import ExecuteInference


class CementBagCounter(ModelWrapper):

    def __init__(self, config, model_config, pubs, device_id):
        super().__init__()
        """
        init function
        """
        self.config = config["config"]
        self.device_id = device_id
        self.rtp = pubs.rtp_write
        self.mongo_logger = MongoLogger()
        self.frame_skip = self.config.get('frame_skip', False)
        model = "data/ACC_v3.pt"
        self.yp = ExecuteInference(weight=model,
                                   gpu=model_config.get("gpu", False),
                                   agnostic_nms=model_config.get("agnostic_nms", True),
                                   iou=model_config.get("iou", 0.2),
                                   confidence=model_config.get("confidence", 0.4))
        self.print_eu_dist = model_config.get('print_eu_dist', 200)
        self.ct1 = CentroidTracker(maxDisappeared=5)
        self.ct2 = CentroidTracker(maxDisappeared=5)
        self.frame_skipping = {
            "skip_current_frame": True,
            "detection_value": None
        }
        self.count = 0
        self.cement_bag = 0
        self.count_suraksha = 0
        self.count_whitecem = 0
        self.count_gold = 0
        self.tracker_list = []
        self.max_age = 3
        self.min_hits = 0
        self.track_id_list = deque([str(i) for i in range(1, 50)])
        self.prev_annotation = []


        self.initial_object_position = None

        self.uncounted_objects = ExpiringDict(max_len=model_config.get("uncounted_obj_length", 50),
                                              max_age_seconds=model_config.get("uncounted_obj_age", 60))
        self.janus_metadata = ExpiringDict(max_age_seconds=120, max_len=1)

    def _pre_process(self, x):
        """
        Do preprocessing here, if any
        :param x: payload
        :return: payload
        """
        return x

    def _post_process(self, x):
        """
        Apply post processing here, if any
        :param x: payload
        :return: payload
        """
        self.rtp.publish(x)  # video stream
        return x

    def send_payload(self, frame, label='CementBagDetected', bg_color="#474520", font_color="#FFFF00", alert_sound=None,
                     message="Cement Bag Detected!"):
        """
        Insert event to Mongo
        :param message:
        :param frame:
        :param label:
        :param bg_color:
        :param font_color:
        :param alert_sound:
        :return: None
        """

        payload = {"deviceId": self.device_id, "message": message,
                   "frame": 'data:image/jpeg;base64,' + base64.b64encode(
                       cv2.imencode('.jpg', frame)[1].tostring()).decode("utf-8"), "activity": label,
                   "bg_color": bg_color, "font_color": font_color, "alert_sound": alert_sound}

        self.mongo_logger.insert_attendance_event_to_mongo(payload)

    def kalman_tracker(
            self,
            bboxs,
            img,
    ):

        z_box = bboxs
        x_box = []

        if len(self.tracker_list) > 0:
            for trk in self.tracker_list:
                x_box.append(trk.box)

        matched, unmatched_dets, unmatched_trks = self.assign_detections_to_trackers(x_box, z_box, iou_thrd=0.03)

        # Deal with matched detections
        if matched.size > 0:
            for trk_idx, det_idx in matched:
                z = z_box[det_idx]
                z = np.expand_dims(z, axis=0).T
                tmp_trk = self.tracker_list[trk_idx]
                tmp_trk.kalman_filter(z)
                xx = tmp_trk.x_state.T[0].tolist()
                xx = [xx[0], xx[2], xx[4], xx[6]]
                x_box[trk_idx] = xx
                tmp_trk.box = xx
                tmp_trk.hits += 1

        # Deal with unmatched detections
        if len(unmatched_dets) > 0:
            for idx in unmatched_dets:
                z = z_box[idx]
                z = np.expand_dims(z, axis=0).T
                tmp_trk = Tracker()  # Create a new tracker
                x = np.array([[z[0], 0, z[1], 0, z[2], 0, z[3], 0]]).T
                tmp_trk.x_state = x
                tmp_trk.predict_only()
                xx = tmp_trk.x_state
                xx = xx.T[0].tolist()
                xx = [xx[0], xx[2], xx[4], xx[6]]
                tmp_trk.box = xx
                tmp_trk.id = self.track_id_list.popleft()  # assign an ID for the tracker

                self.tracker_list.append(tmp_trk)
                x_box.append(xx)

        # Deal with unmatched tracks
        if len(unmatched_trks) > 0:
            for trk_idx in unmatched_trks:
                tmp_trk = self.tracker_list[trk_idx]
                tmp_trk.no_losses += 1
                tmp_trk.predict_only()
                xx = tmp_trk.x_state
                xx = xx.T[0].tolist()
                xx = [xx[0], xx[2], xx[4], xx[6]]
                tmp_trk.box = xx
                x_box[trk_idx] = xx

        # The list of tracks to be annotated
        good_tracker_list = []
        objects = []
        boxs = []
        for trk in self.tracker_list:
            if (trk.hits >= self.min_hits) and (trk.no_losses <= self.max_age):
                good_tracker_list.append(trk)
                x_cv2 = trk.box
                left, top, right, bottom = x_cv2[1], x_cv2[0], x_cv2[3], x_cv2[2]
                centroid = [int(left + ((right - left) / 2)), bottom]
                objects.append([int(trk.id), centroid])
                boxs.append(x_cv2)

        deleted_tracks = filter(lambda _x: _x.no_losses > self.max_age, self.tracker_list)

        for trk in deleted_tracks:
            self.track_id_list.append(trk.id)

        self.tracker_list = [x for x in self.tracker_list if x.no_losses <= self.max_age]
       # print("object is ", str(objects))

        return img, objects, boxs

    @staticmethod
    def assign_detections_to_trackers(
            trackers,
            detections,
            iou_thrd=0.3,
    ):
        """
        From current list of trackers and new detections, output matched detections,
        un matched trackers, unmatched detections.
        """
        iou_mat = np.zeros((len(trackers), len(detections)), dtype=np.float32)
        for t, trk in enumerate(trackers):
            for d, det in enumerate(detections):
                iou_mat[t, d] = box_iou2(trk, det)

        matched_idx = linear_assignment(-iou_mat)

        unmatched_trackers, unmatched_detections = [], []
        for t, trk in enumerate(trackers):
            if t not in matched_idx[:, 0]:
                unmatched_trackers.append(t)

        for d, det in enumerate(detections):
            if d not in matched_idx[:, 1]:
                unmatched_detections.append(d)

        matches = []

        for m in matched_idx:
            if iou_mat[m[0], m[1]] < iou_thrd:
                unmatched_trackers.append(m[0])
                unmatched_detections.append(m[1])
            else:
                matches.append(m.reshape(1, 2))

        if len(matches) == 0:
            matches = np.empty((0, 2), dtype=int)
        else:
            matches = np.concatenate(matches, axis=0)

        return matches, np.array(unmatched_detections), np.array(unmatched_trackers)



    def get_line_coordinates(self):
        """
        Get the line coordinates from the deployment JSON
        """
        if not self.janus_metadata.get('metadata'):
            self.janus_metadata['metadata'] = get_extra_fields(self.device_id)

        _coordinates = [self.janus_metadata['metadata'].get(coordinate_key) for coordinate_key in
                        JanusDeploymentConstants.LINE_COORDINATES]
        _alignment = self.janus_metadata['metadata'].get(JanusDeploymentConstants.ALIGNMENT_KEY)\
        # _coordinates = [550, 200, 555, 1100]
        #
        # _alignment = "vertical"
        return _alignment, _coordinates

    def line_point_position(self, point):
        """
        Get the position of point w.r.t. the line
        :param point: point to be compared
        :return: boolean
        """
        _alignment, line_coordinates = self.get_line_coordinates()

        assert len(line_coordinates) == 4, "Line coordinates variable is invalid"
        assert len(point) == 2, "Point variable is invalid"

        _slope = (line_coordinates[3] - line_coordinates[1]) / (line_coordinates[2] - line_coordinates[0])
        _point_equation_value = point[1] - line_coordinates[1] - _slope * (point[0] - line_coordinates[0])
        if _point_equation_value > 0:
            return True
        else:
            return False

    def validate_point_position(self, point):
        """
        Validate the position of the point w.r.t. the line
        :param point: centroid
        :return: bool
        """
        _alignment, line_coordinates = self.get_line_coordinates()
        assert _alignment in [JanusDeploymentConstants.VERTICAL, JanusDeploymentConstants.HORIZONTAL], \
            "Invalid alignment variable"
        if _alignment == JanusDeploymentConstants.VERTICAL:
        # _alignment, line_coordinates = self.get_line_coordinates()
        # assert _alignment in ["horizontal", "vertical"], \
        #     "Invalid alignment variable"
        # print(point)
        # if _alignment == "vertical":
            line_y2 = line_coordinates[3]
            line_y1 = line_coordinates[1]
            if line_y1 < point[1] < line_y2 or line_y2 < point[1] < line_y1:
                return True
            else:
                return False
        else:
            line_x2 = line_coordinates[2]
            line_x1 = line_coordinates[0]
            if line_x1 < point[0] < line_x2 or line_x2 < point[0] < line_x1:

                return True
            else:
                return False

    def update_bag_count(self, frame, detection_objects, class_name, detections):
        """
        Maintains the bag counts
        :param frame: image
        :param detection_objects: detection object having object id and centroids
        """
        #for class_name, (objectID, centroid) in zip(classes, detection_objects):
        for (object_id, det) in zip(detection_objects, detections):
            centroid = object_id[1]
            object_id = object_id[0]
            logger.debug(detections)
            #print(object_id)
            frame = draw_circles_on_frame(frame, centroid, radius=10, color=(0, 0, 255),
                                          thickness=-1)
            if self.validate_point_position(centroid):
                logger.debug("centroid detected")
                #if self.validate_point_position(centroid):

                # # if not isinstance(self.count, int):
                # #     logger.debug("Initializing the count variable")
                # print("again entering")
                # # Initializing the bag count
                # if (class_name == "acc_gold"):
                #     if not isinstance(self.count_gold, int):
                #         logger.debug("Initializing the count variable")
                #         self.count_gold = 0
                # elif (class_name == "acc_suraksha"):
                #     if not isinstance(self.count_suraksha, int):
                #         logger.debug("Initializing the count variable")
                #         self.count_suraksha = 0
                # elif (class_name == "acc_buildcem"):
                #     if not isinstance(self.count_whitecem, int):
                #         logger.debug("Initializing the count variable")
                #         self.count_whitecem = 0

                if not isinstance(self.initial_object_position, bool):
                    logger.debug("Initializing the initial object position")
                    #self.initial_object_position = self.line_point_position(point=centroid)
                    self.initial_object_position = True
                    logger.debug(self.initial_object_position)

                _point_position = self.line_point_position(point=centroid)
                #print("object ID is : ", str(objectID))
                logger.debug(self.uncounted_objects)

                # Check point in the same side as the initial object
                if _point_position == self.initial_object_position:
                    logger.debug("same side only")
                    #print(class_name)
                    # Check the object is not already counted
                    if object_id not in self.uncounted_objects:
                        self.uncounted_objects[object_id] = centroid


                elif object_id in self.uncounted_objects:
                    self.uncounted_objects.pop(object_id, None)
                    if ("acc_gold" in class_name):
                        self.count_gold += 1
                        mrp_result = self.distances(detections)
                        if mrp_result:
                            self.send_payload(resize_to_64_64(frame=frame),
                                              message='ACC GOLD Bag Detected: Print Detected!')
                            logger.info(f"Count: {self.count_gold}, Print Found: True")
                        else:
                            self.send_payload(resize_to_64_64(frame=frame),
                                              message='ACC GOLD Bag Detected: Print Missing!')
                            logger.info(f"Count: {self.count_gold}, Print Found: False")



                    elif ("acc_suraksha_plus" in class_name):
                        self.count_suraksha += 1
                        logger.debug(self.count_suraksha)
                        mrp_result = self.distances(detections)
                        if mrp_result:
                            self.send_payload(resize_to_64_64(frame=frame),
                                              message='ACC SURAKSHA PLUS Bag Detected: Print Detected!')
                            logger.info(f"Count: {self.count_suraksha}, Print Found: True")
                        else:
                            self.send_payload(resize_to_64_64(frame=frame),
                                              message='ACC SURAKSHA PLUS Bag Detected: Print Missing!')
                            logger.info(f"Count: {self.count_suraksha}, Print Found: False")

                    elif ("ambuja_whitecem" in class_name):
                        self.count_whitecem += 1
                        mrp_result = self.distances(detections)
                        if mrp_result:
                            self.send_payload(resize_to_64_64(frame=frame),
                                              message='PPC White Bag Detected: Print Detected!')
                            logger.info(f"Count: {self.count_whitecem}, Print Found: True")
                        else:
                            self.send_payload(resize_to_64_64(frame=frame),
                                              message='PPC White Bag Detected: Print Missing!')
                            logger.info(f"Count: {self.count_whitecem}, Print Found: False")

                    frame = draw_circles_on_frame(frame, centroid['centroid'], radius=10, color=(0, 255, 0),
                                                  thickness=-1)
                    if centroid['has_print']:
                        self.send_payload(resize_to_64_64(frame=frame), message='Print Detected!')
                        logger.info(f"Count: {self.count}, Print Found: True")
                    else:
                        self.send_payload(resize_to_64_64(frame=frame), message='Print Missing!')
                        logger.info(f"Count: {self.count}, Print Found: False")
                else:
                    frame = draw_circles_on_frame(frame, centroid['centroid'], radius=10, color=(0, 255, 0),
                                                  thickness=-1)

        # count_text_gold = "ACC_GOLD: " + str(self.count_gold)
        # count_text_suraksha = "ACC_SURAKSHA_PLUS: " + str(self.count_suraksha)
        # count_text_whitecem = "ACC_WHITE_CEM: " + str(self.count_whitecem)
        # cv2.putText(frame, count_text_gold, (1300, 200), cv2.FONT_HERSHEY_SIMPLEX, 1, (255, 255, 0), 3,
        #             cv2.LINE_AA)
        # cv2.putText(frame, count_text_suraksha, (1300, 400), cv2.FONT_HERSHEY_SIMPLEX, 1, (255, 255, 0), 3,
        #             cv2.LINE_AA)
        # cv2.putText(frame, count_text_whitecem, (1300, 600), cv2.FONT_HERSHEY_SIMPLEX, 1, (255, 255, 0), 3,
        #             cv2.LINE_AA)
        return frame

    def draw_line_over_image(self, frame, color=(255, 255, 255)):
        """
        Draws line over the counting line
        :param frame: frame for
        :param color:
        :return:
        """
        _alignment, line_coordinates = self.get_line_coordinates()
        assert len(line_coordinates) == 4, "Line coordinates variable is invalid"

        # return cv2.line(frame, (line_coordinates[0], line_coordinates[1]), (line_coordinates[2], line_coordinates[3]),
        #                 color, 3)

        self.drawline(frame, (line_coordinates[0], line_coordinates[1]), (line_coordinates[2],
                                                                          line_coordinates[3]), color, thickness=3)
        return frame

    @staticmethod
    def drawline(img, pt1, pt2, color, thickness=1, style='dotted', gap=20):
        dist = ((pt1[0] - pt2[0]) ** 2 + (pt1[1] - pt2[1]) ** 2) ** .5
        pts = []
        for i in np.arange(0, dist, gap):
            r = i / dist
            x = int((pt1[0] * (1 - r) + pt2[0] * r) + .5)
            y = int((pt1[1] * (1 - r) + pt2[1] * r) + .5)
            p = (x, y)
            pts.append(p)

        if style == 'dotted':
            for p in pts:
                cv2.circle(img, p, thickness, color, -1)
        else:
            s = pts[0]
            e = pts[0]
            i = 0
            for p in pts:
                s = e
                e = p
                if i % 2 == 1:
                    cv2.line(img, s, e, color, thickness)
                i += 1

    def distances(self, detections):
        mrp_cord = list()
        cem_bag_cord = list()
        for det in detections:
            if(det["class"] == "mrp"):
                mrp_cord.append(det["points"])
            else:
                cem_bag_cord.append(det["points"])
        for c_cord in cem_bag_cord:
            for m_cord in mrp_cord:

                if (m_cord[0] > c_cord[0] and m_cord[0] < c_cord[2] and
                        m_cord[1] > c_cord[1] and m_cord[1] < c_cord[3]):
                    logger.debug("print is detected")
                    #cv2.waitKey(0)
                    return True
                else:
                    return False


    def inference(
            self,
            frame,
            classes,

    ):
        dets = self.yp.predict(frame)
        class_name = list()
        bboxs = []

        if dets:
            for i in dets:
                if(i["class"] in classes):
                    class_name.append(i["class"])
                    #cv2.rectangle(frame, (i["points"][0], i["points"][1]), (i["points"][2], i["points"][3]), (255, 255, 0), 2)
                    bboxs.append([i["points"][1], i["points"][0], i["points"][3], i["points"][2]])

#        frame = cv2.rectangle(frame, (bboxs[0][0], bboxs[0][1]), (bboxs[0][2], bboxs[0][3]),(255, 255, 0) , 2)
        return bboxs, frame, dets, class_name

    def _predict(self, obj):
        class_list = ["acc_gold", "acc_suraksha_plus", "ambuja_whitecem"]
        mrp = ["mrp"]
        try:
            frame = obj['frame']

            if self.frame_skip:
                if not self.frame_skipping["skip_current_frame"]:
                    dets = self.yp.predict(frame)
                    self.frame_skipping["detection_value"] = dets
                    self.frame_skipping["skip_current_frame"] = True
                else:
                    dets = self.frame_skipping["detection_value"]
                    self.frame_skipping["skip_current_frame"] = False
            else:
                dets, frame, _dets, class_name = self.inference(frame, class_list)


                #dets_mrp, frame_mrp, _dets_mrp, class_name_mrp = self.inference(frame, mrp)

            frame = self.draw_line_over_image(frame)
            # if [True for e in dets if e['class'] == 'cement_bag']:


            #if dets:
            frame, objects, boxs = self.kalman_tracker(dets, frame)
            print("PRINTING KALMAN OUTPUT")
            print(objects)
            print(boxs)
            for box in boxs:
                cv2.rectangle(frame, (box[1], box[0]), (box[3], box[2]), (255, 0, 0), 2)

            #objects,classes_cement, frame = self.track_bags(self.ct1, dets, frame, class_list)
            #_,classes, frame = self.track_bags(self.ct2, _dets, frame, mrp)
            #frame, _, box_mrp = self.kalman_tracker(dets_mrp, frame)
            frame = self.update_bag_count(frame=frame, detection_objects=objects, class_name = class_name, detections = _dets)
            # cv2.imshow("output is ", cv2.resize(frame, (900, 600)))
            # cv2.waitKey(1)
            # print("******")
            # print(objects)
            # print(_)

            # self.distances(objects, _)
            logger.debug("self.uncounted_objects --> {}".format(self.uncounted_objects))
            # for each in dets:
            #     color = (255, 255, 0)
            #     class_n = "Cement Bag"
            #
            #     if each['class'] == 'label':
            #         color = (0, 255, 0)
            #         class_n = "Printing Detected!"
            #     cv2.rectangle(frame, (each['points'][0], each['points'][1]), (each['points'][2], each['points'][3]),
            #                   color, 2)
            #     cv2.putText(frame, class_n, (each['points'][2], each['points'][1]), cv2.FONT_HERSHEY_SIMPLEX,
            #                 1, color, 2, cv2.LINE_AA)

            obj['frame'] = cv2.resize(frame, (self.config.get('FRAME_WIDTH'), self.config.
                                              get('FRAME_HEIGHT')))
        except Exception as e:
            logger.exception(f"Error: {e}", exc_info=True)
            obj['frame'] = cv2.resize(obj['frame'], (self.config.get('FRAME_WIDTH'), self.config.get('FRAME_HEIGHT')))

        return obj
